/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Test application.                                      *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include <windows.h>
#include <stdio.h>                          /* FILE */
#include <gl/gl.h>
#include <gl/glut.h>

#include "dslworld.h"                       /* scene manager */
#include "dslmodel.h"                       /* model drawing */
#include "dslutil.h"                        /* X,Y,Z */
#include "dslofscr.h"                       /* offscreen drawing */
#include "dslplanr.h"                       /* planar texture */
#include "dslproj.h"                        /* projective texture */
#include "dslvolum.h"                       /* shadow volume */
#include "dslshmap.h"                       /* shadow maps */
#include "dslprmap.h"                       /* priority maps */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Camera camera =                      /* Viewing camera */
{
 {-6.0, 103.0, 93.0, 1.0},
 {50.0, 5.0, 0.0},

 60.0,
 256,256,
 1.0,2000.0,
};

struct Light light=                         /* single lightsource */
{
 {0.15, 0.15, 0.15},
 {0.8, 0.8, 0.8},
 {1.0, 1.0, 1.0},
 {0.0, 60.0, 0.0, 1.0},

 1.0 ,0 ,0                                  /* no attenuation */
};

struct Plane plane=                         /* plane equation */
{
 {0.0, -100.0, 0.0, 1.0},
 {0.0, 1.0, 0.0, 1.0}
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Model *mod,*pln,*spr;                /* model pointers */
struct ModelInstance modi,plni,spri;        /* model instances */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

int drawState=0;

/**********************************************************\
 * Render a frame.                                        *
\**********************************************************/

void render(void)
{
 struct ModelInstance* modlist[]={&modi,&spri,&plni};

 cameraEnable(&camera);                     /* set the camera */
 lightEnable(&light,0);                     /* and the light source */

 switch(drawState)
 {
  case 0:                                   /* draw planar model */
  {
   /* Draw the receiver (plane) first so that stencil buffer
    * could be updated.
    */
   planarShadowDrawReceiver(&plni); 
   /* For each model casting shadow on to a plane from a light source,
    * draw the model twice first as is, second squished to that 
    * plane.
    */
   planarShadowDrawScene(modlist,2,&plane,light.position);
  }
  break;

  case 1:                                   /* draw projective texture */
  {
   /* Draw the texture to be projected onto castees and draw the
    * model of the caster.
    */
   projectiveTextureDrawCaster(&modi,light.position);
   /* Draw all the castees using the projective texture computed
    * above.
    */
   projectiveTextureDrawScene(modlist+1,2,light.position);
  }
  break;

  case 2:                                   /* draw shadow volumes */
  {
   /* Draw the scene without any shadows.
    */
   shadowVolumeDrawScene(modlist,3);
   /* Draw all shadow volumes and produce the shadow map
    * using stencil buffer.
    */ 	
   shadowVolumeDrawMap(modlist,2,light.position,200.0);
   /* Blend the image of the scene and the shadow map.
    */
   shadowVolumeBlend();
  }
  break;

  case 3:                                   /* draw shadow buffer */
  {
   /* Draw the image from the position of the light source using
    * attenuated illumination. Map this onto the scene. Blend
    * this with an image of the scene without mapping but with
    * the attenuated light as before. This produces the shadow
    * map.
    */
   shadowMapDrawMap(modlist,3,light.position,plni.position,60.0);  
   /* Draw the scene in its actual colors and subtractively blend
    * with this image of the shadow map computed above.
    */
   shadowMapDrawScene(modlist,3);
  }
  break;


  case 4:                                   /* draw priority buffer */
  {
   /* Draw the texture from the position of the light source using
    * distinct increasing colors for each model. Map this onto the
    * scene. Blend this with an image of the scene without mapping
    * but with same coloring of models. This produces the shadow
    * map.
    */
   priorityMapDrawMap(modlist,3,light.position,plni.position,60.0);  
   /* Draw the scene in its actual colors and subtractively blend
    * with the image of the shadow map computed above.
    */
   priorityMapDrawScene(modlist,3);
  }
  break;
 }

 lightDraw(&light);                         /* mark position of the light source */

 glutSwapBuffers();                         /* display the scene */
}

/**********************************************************\
 * Respond to special keyboard events -- arrows.          *
\**********************************************************/

void special(int key, int x, int y)
{
 switch(key)
 {
  case GLUT_KEY_UP: cameraRotate(&camera,X,5.0);
       break;       
  case GLUT_KEY_DOWN: cameraRotate(&camera,X,-5.0);
       break;                                   
  case GLUT_KEY_LEFT: cameraRotate(&camera,Y,5.0);
       break;                                   
  case GLUT_KEY_RIGHT: cameraRotate(&camera,Y,-5.0);
       break;
  case GLUT_KEY_PAGE_UP: cameraAdvance(&camera,5.0);
       break;
  case GLUT_KEY_PAGE_DOWN: cameraAdvance(&camera,-5.0);
       break;
 }
 
 glutPostRedisplay();
}

/**********************************************************\
 * Respond to keyboard events -- keys.                    *
\**********************************************************/

void keyboard(unsigned char key, int x, int y)
{
 drawState=(drawState+1)%5;                 /* 5 different algorithms */

 glutPostRedisplay();
}

/**********************************************************\
 * Respond to window resizing.                            *
\**********************************************************/

void resize(int w, int h)
{
 cameraSetViewport(&camera,w,h);
}

/**********************************************************\
 * Initialize and run rendering loop.                     *
\**********************************************************/

int main(int argc, char* argv[])
{
 glutInit(&argc, argv);
 glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH | GLUT_STENCIL);
 glutCreateWindow("SDSL2");
 glutReshapeWindow(camera.sizeX, camera.sizeY);
 offscreenInit(camera.sizeX, camera.sizeY);
	
 glutReshapeFunc(resize);                   /* setup callback functions */
 glutSpecialFunc(special);
 glutKeyboardFunc(keyboard);
 glutDisplayFunc(render);

 glEnable(GL_DEPTH_TEST);                   /* use Z buffer */
 glClearColor(0.0f, 0.0f, 0.0f, 1.0f );

 {                                          /* read the models */
  FILE* f=fopen("models/teapot.m","r");
  mod=modelCreate(f);
  modelScale(mod,55.0);
  fclose(f);
  modelInstanceCreate(&modi,mod);
  modelInstanceSetPosition(&modi,0,0,0);
  modelInstanceSetOrientation(&modi,0,-40,0);

  f=fopen("models/sphere.m","r");
  spr=modelCreate(f);
  modelScale(spr,20.0);
  fclose(f);
  modelInstanceCreate(&spri,spr);
  modelInstanceSetPosition(&spri,40,-55,5);
  modelInstanceSetOrientation(&spri,0,0,0);

  f=fopen("models/plane.m","r");
  pln=modelCreate(f);
  fclose(f);
  modelInstanceCreate(&plni,pln);
  modelInstanceSetPosition(&plni,0,-100.0,0);
  modelInstanceSetOrientation(&plni,0,0,0);
 }

 glutSwapBuffers();                         /* fixes strange stencil bug? */
 glutMainLoop();                            /* run the event loop */

 offscreenDeinit();
                                            /* free texture memory */
 return(0);
}

/**********************************************************/
