/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Scene manager.                                         *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include "dslworld.h"
#include "dslutil.h"                        /* matrixMultiply etc. */
#include <gl/glu.h>                         /* gluPerspective */
#include <gl/glut.h>                        /* glutSolidSphere */
#include <string.h>                         /* strcmp */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Light* pCurrentLight;                /* light that was last enabled */
int nCurrentLight;                          /* at which position was it */

/**********************************************************\
 * Construct a vector from two points.                    *
\**********************************************************/

void cameraSetViewport(struct Camera* pCamera, int sx, int sy)
{
 pCamera->sizeX=sx;                         /* current dimensions of viewport */
 pCamera->sizeY=sy;
}

/**********************************************************\
 * Construct a vector from two points.                    *
\**********************************************************/

void cameraAdvance(struct Camera* pCamera, float distance)
{
 GLfloat modelView[4][4];
 GLfloat arg[4];
 GLfloat res[4];

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();                            /* preserve current state */

 glLoadIdentity();
 glTranslatef(pCamera->position[X],pCamera->position[Y],pCamera->position[Z]);
 glRotatef(-pCamera->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(-pCamera->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(-pCamera->orientation[Z],0.0f,0.0f,1.0f);

 arg[X]=arg[Y]=0.0;
 arg[Z]=-distance;
 arg[W]=1.0;
                                            /* extract view to world (!) */
 glGetFloatv(GL_MODELVIEW_MATRIX,(GLfloat*)modelView);

 matrixMultiply(res,modelView,arg);         /* next camera position in world */

 pCamera->position[X]=res[X];
 pCamera->position[Y]=res[Y];
 pCamera->position[Z]=res[Z];

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/**********************************************************\
 * Rotate the camera around specified axis by a given     *
 * distance.                                              *
/**********************************************************/

void cameraRotate(struct Camera* pCamera, int axis, float distance)
{
 pCamera->orientation[axis]+=distance;
}

/**********************************************************\
 * Construct a vector from two points.                    *
\**********************************************************/

void cameraEnable(struct Camera* pCamera)
{
 glViewport(0, 0, pCamera->sizeX, pCamera->sizeY);

 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 glRotatef(pCamera->orientation[Z],0.0f,0.0f,1.0f);
 glRotatef(pCamera->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pCamera->orientation[Y],0.0f,1.0f,0.0f);
 glTranslatef(-pCamera->position[X],-pCamera->position[Y],-pCamera->position[Z]);

 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();                          /* projection changes (aspect) */
 gluPerspective(pCamera->fovY, (GLfloat)pCamera->sizeX/(GLfloat)pCamera->sizeY,
                pCamera->nearZ, pCamera->farZ);
}

/**********************************************************\
 * Pass light parameters.                                 *
\**********************************************************/

void lightEnable(struct Light* pLight, int hwLightId )
{
 GLenum lid[4]={GL_LIGHT0,GL_LIGHT1,GL_LIGHT2,GL_LIGHT3};

 pCurrentLight=pLight;
 nCurrentLight=hwLightId;

 glLightfv(lid[hwLightId],GL_AMBIENT,pLight->ambient);
 glLightfv(lid[hwLightId],GL_DIFFUSE,pLight->diffuse);
 glLightfv(lid[hwLightId],GL_SPECULAR,pLight->specular);

 glLightf(lid[hwLightId],GL_CONSTANT_ATTENUATION,pLight->constantAttenuation);
 glLightf(lid[hwLightId],GL_LINEAR_ATTENUATION,pLight->linearAttenuation);
 glLightf(lid[hwLightId],GL_QUADRATIC_ATTENUATION,pLight->quadraticAttenuation);

 glLightfv(lid[hwLightId],GL_POSITION,pLight->position);
 glEnable(lid[hwLightId]);                  /* create a lightsource */
}

/**********************************************************\
 * Draw a light as a small solid sphere.                  *
\**********************************************************/

void lightDraw(struct Light* pLight)
{
 glDisable(GL_LIGHTING);   
 glColor3f(1.0, 1.0, 1.0);

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pLight->position[X],pLight->position[Y],pLight->position[Z]);
 glutSolidSphere(LIGHT_DRAW_DIAMETER,LIGHT_DRAW_SPLITS,LIGHT_DRAW_SPLITS);

 glPopMatrix();
}

/**********************************************************\
 * Read material from an external file.                   *
\**********************************************************/

struct Material* materialCreate(FILE* f)
{
 char sTag[64];
 struct Material* pMaterial;

 pMaterial=(struct Material*)malloc(sizeof(struct Material));
 if(pMaterial==NULL) return(NULL);

 fscanf(f,"%s %f %f %f",sTag,&pMaterial->ambient[R],&pMaterial->ambient[G],&pMaterial->ambient[B]);
 if(strcmp(sTag,"AMBIENT")!=0) return(NULL);

 fscanf(f,"%s %f %f %f",sTag,&pMaterial->diffuse[R],&pMaterial->diffuse[G],&pMaterial->diffuse[B]);
 if(strcmp(sTag,"DIFFUSE")!=0) return(NULL);

 fscanf(f,"%s %f %f %f",sTag,&pMaterial->specular[R],&pMaterial->specular[G],&pMaterial->specular[B]);
 if(strcmp(sTag,"SPECULAR")!=0) return(NULL);

 fscanf(f,"%s %f",sTag,&pMaterial->shininess);
 if(strcmp(sTag,"SHININESS")!=0) return(NULL);

 return(pMaterial);
}

/**********************************************************\
 * Delete the material.                                   *
\**********************************************************/

void materialDestroy(struct Material* pMaterial)
{
 free(pMaterial);
}

/**********************************************************\
 * Enable a particular material.                          *
\**********************************************************/

void materialEnable(struct Material* pMaterial)
{
 glMaterialfv(GL_FRONT,GL_AMBIENT,pMaterial->ambient);
 glMaterialfv(GL_FRONT,GL_DIFFUSE,pMaterial->diffuse);
 glMaterialfv(GL_FRONT,GL_SPECULAR,pMaterial->specular);
 glMateriali(GL_FRONT,GL_SHININESS,pMaterial->shininess);
}

/**********************************************************/
  