/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Drawing of shadow volumes.                             *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include <windows.h>
#include <gl/glu.h>                         /* gluOrtho2D */
#include "dslvolum.h"                       /* struct Model */
#include "dslutil.h"                        /* scalarProduct */
#include "dslofscr.h"                       /* offscreenSizeX and Y */

/**********************************************************\
 * Draw the scene (to create Z buffer).                   *
\**********************************************************/

void shadowVolumeDrawScene(struct ModelInstance** pModelInstances, int nModelInstances)
{
 int i;

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 for(i=0;i<nModelInstances;i++)
  modelInstanceDrawSolid(pModelInstances[i]);

 glReadBuffer(GL_BACK);
 glAccum(GL_LOAD,1.0);                      /* as is */
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw shadow volume of a single model.                  *
\**********************************************************/

void shadowVolumeDraw(struct ModelInstance* pModelInstance, GLfloat* lightPosition, GLfloat length)
{
 int i;
 GLfloat endPosition0[4],endPosition1[4];
 GLfloat modelLightPosition[4];
 GLfloat worldModel[4][4];
 struct Model* pModel=pModelInstance->pModel;

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glLoadIdentity();
 glRotatef(-pModelInstance->orientation[Z],0.0f,0.0f,1.0f);
 glRotatef(-pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(-pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glTranslatef(-pModelInstance->position[X],-pModelInstance->position[Y],-pModelInstance->position[Z]);

 glGetFloatv(GL_MODELVIEW_MATRIX,(GLfloat*)worldModel);
 matrixMultiply(modelLightPosition,worldModel,lightPosition);

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glBegin(GL_TRIANGLES);
 for(i=0;i<pModel->nEdges;i++)
 {
  int cws,ccws;

  if(pModel->pEdges[i].ccwt==NULL) { cws=0; ccws=1; }
  else
  {
   GLfloat vcwt[4],vccwt[4];

   vectorConstruct(vcwt,pModel->pEdges[i].cwt->m,modelLightPosition);
   vectorConstruct(vccwt,pModel->pEdges[i].ccwt->m,modelLightPosition);

   cws=vectorScalarProduct(pModel->pEdges[i].cwt->n,vcwt)>0;
   ccws=vectorScalarProduct(pModel->pEdges[i].ccwt->n,vccwt)>0; 
  }
 
  if(cws!=ccws)                             /* if a silouette edge */
  {
   vectorConstruct(endPosition0,pModel->pEdges[i].pv[0]->v,modelLightPosition);
   vectorConstruct(endPosition1,pModel->pEdges[i].pv[1]->v,modelLightPosition);

   vectorNormalize(endPosition0);
   vectorNormalize(endPosition1);

   vectorScale(endPosition0,endPosition0,length);
   vectorScale(endPosition1,endPosition1,length);

   vectorAdd(endPosition0,endPosition0,pModel->pEdges[i].pv[0]->v);
   vectorAdd(endPosition1,endPosition1,pModel->pEdges[i].pv[1]->v);

   if(cws==1)                               /* extrude the edge */
   {
    glVertex3fv(pModel->pEdges[i].pv[0]->v);
    glVertex3fv(endPosition1);
    glVertex3fv(endPosition0);
	
    glVertex3fv(pModel->pEdges[i].pv[0]->v);
    glVertex3fv(pModel->pEdges[i].pv[1]->v);
    glVertex3fv(endPosition1);
   }
   else
   {
    glVertex3fv(pModel->pEdges[i].pv[0]->v);
    glVertex3fv(endPosition0);
    glVertex3fv(endPosition1);

    glVertex3fv(pModel->pEdges[i].pv[0]->v);
    glVertex3fv(endPosition1);
    glVertex3fv(pModel->pEdges[i].pv[1]->v);
   }
  }
 }
 glEnd();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/**********************************************************\
 * Draw shadow volumes of models (using Z buffer)         *
 * to produce shadow stencil.                             *
\**********************************************************/

void shadowVolumeDrawMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightDirection, GLfloat length)
{
 int i;

 glDisable(GL_LIGHTING);

 glClearStencil(0xf);   
 glClear(GL_STENCIL_BUFFER_BIT);
 
 glStencilFunc(GL_ALWAYS,0,0);
 glEnable(GL_STENCIL_TEST);
 glEnable(GL_CULL_FACE);
 glDepthMask(GL_FALSE);
 glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);

 for(i=0;i<nModelInstances;i++)
 {
  glStencilOp(GL_KEEP,GL_KEEP,GL_DECR);
  glCullFace(GL_BACK);                      /* only front polygons drawn */
  shadowVolumeDraw(pModelInstances[i],lightDirection,length);

  glStencilOp(GL_KEEP,GL_KEEP,GL_INCR);
  glCullFace(GL_FRONT);                     /* only back polygons drawn */
  shadowVolumeDraw(pModelInstances[i],lightDirection,length);
 }

 glDisable(GL_CULL_FACE);
 glDisable(GL_STENCIL_TEST);
 glDepthMask(GL_TRUE);
 glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
}

/**********************************************************\
 * Draw the scene through the stencil
\**********************************************************/

void shadowVolumeBlend(void)
{
 glClear(GL_COLOR_BUFFER_BIT);
 glEnable(GL_STENCIL_TEST);
 glDisable(GL_DEPTH_TEST);
 glStencilFunc(GL_NOTEQUAL,0xf,0xf);
 glStencilOp(GL_KEEP,GL_KEEP,GL_KEEP);

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glLoadIdentity();

 glMatrixMode(GL_PROJECTION);
 glPushMatrix();
 glLoadIdentity();

 gluOrtho2D(0,0,offscreenSizeX-1,offscreenSizeY-1);

 glDisable(GL_LIGHTING);
 glBegin(GL_QUADS);
 glColor3f(1.0f,1.0f,1.0f);
 glVertex2f(-1.0f,-1.0f);
 glVertex2f(-1.0f,1.0f);
 glVertex2f(1.0f,1.0f);
 glVertex2f(1.0f,-1.0f);
 glEnd();
 
 glMatrixMode(GL_PROJECTION);
 glPopMatrix();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();

 glReadBuffer(GL_BACK);
 glAccum(GL_ACCUM,-1.0f);
 glAccum(GL_RETURN,1.0);                    /* subtract shadow map */
 glDisable(GL_STENCIL_TEST);
 glEnable(GL_DEPTH_TEST);
}

/**********************************************************/
