/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library                  *
 *                                                        *
 * Vector utility functions.                              *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include <math.h>                           /* sqrt */

#include "dslutil.h"

/**********************************************************\
 * Construct a vector from two points                     *
\**********************************************************/

GLfloat* vectorConstruct(GLfloat* vec, GLfloat* a, GLfloat* b)
{
 vec[X]=a[X]-b[X]; vec[Y]=a[Y]-b[Y]; vec[Z]=a[Z]-b[Z];
 vec[W]=1.0f;

 return(vec);
}

/**********************************************************\
 * Zero a vector.                                         *
\**********************************************************/

GLfloat* vectorZero(GLfloat* v)
{
 v[X]=v[Y]=v[Z]=0.0f;
 v[W]=1.0f;

 return(v);
}

/**********************************************************\
 * Assign a value of one vector to another.               *
\**********************************************************/

GLfloat* vectorCopy(GLfloat* res, GLfloat* arg)
{
 res[X]=arg[X]; res[Y]=arg[Y]; res[Z]=arg[Z];
 res[W]=1.0f;

 return(res);
}

/**********************************************************\
 * Sum of two vectors.                                    *
\**********************************************************/

GLfloat* vectorAdd(GLfloat* sum, GLfloat* a, GLfloat* b)
{
 sum[X]=a[X]+b[X]; sum[Y]=a[Y]+b[Y]; sum[Z]=a[Z]+b[Z];
 sum[W]=1.0f;

 return(sum);
}

/**********************************************************\
 * Difference of two vectors.                             *
\**********************************************************/

GLfloat* vectorSub(GLfloat* dif, GLfloat* a, GLfloat* b)
{
 dif[X]=a[X]-b[X]; dif[Y]=a[Y]-b[Y]; dif[Z]=a[Z]-b[Z];
 dif[W]=1.0f;

 return(dif);
}

/**********************************************************\
 * Scale of a vector.                                     *
\**********************************************************/

GLfloat* vectorScale(GLfloat* sv, GLfloat* v, GLfloat scl)
{
 sv[X]=v[X]*scl; sv[Y]=v[Y]*scl; sv[Z]=v[Z]*scl;
 sv[W]=1.0f;

 return(sv);
}

/**********************************************************\
 * Scalar product of two vectors.                         *
\**********************************************************/

GLfloat vectorScalarProduct(GLfloat* a, GLfloat* b)
{
 return(a[X]*b[X]+a[Y]*b[Y]+a[Z]*b[Z]);
}

/**********************************************************\
 * Cross product of two vectors.                          *
\**********************************************************/

GLfloat* vectorCrossProduct(GLfloat* cross, GLfloat* a, GLfloat* b)
{
 cross[X]=a[Y]*b[Z]-a[Z]*b[Y];
 cross[Y]=a[Z]*b[X]-a[X]*b[Z];
 cross[Z]=a[X]*b[Y]-a[Y]*b[X];
 cross[W]=1.0f;

 return(cross);
}

/**********************************************************\
 * Length of a vector.                                    *
\**********************************************************/

GLfloat vectorLength(GLfloat* a)
{
 return((GLfloat)sqrt(a[X]*a[X]+a[Y]*a[Y]+a[Z]*a[Z]));
}

/**********************************************************\
 * Normalizing a vector.                                  *
\**********************************************************/

void vectorNormalize(GLfloat* norm)
{
 GLfloat lng;

 lng=(GLfloat)sqrt(norm[X]*norm[X]+norm[Y]*norm[Y]+norm[Z]*norm[Z]);

 norm[X]/=lng; norm[Y]/=lng; norm[Z]/=lng;
}

/**********************************************************\
 * Finding a normal given three points.                   *
\**********************************************************/

GLfloat* vectorNormal(GLfloat* norm, GLfloat* a, GLfloat* b, GLfloat* c)
{
 GLfloat lng;
 GLfloat u[4], v[4];

 u[X]=b[X]-a[X]; u[Y]=b[Y]-a[Y]; u[Z]=b[Z]-a[Z];
 v[X]=c[X]-b[X]; v[Y]=c[Y]-b[Y]; v[Z]=c[Z]-b[Z];

 norm[X]=u[Y]*v[Z]-u[Z]*v[Y];               /* primitive functions */
 norm[Y]=u[Z]*v[X]-u[X]*v[Z];               /* don't call one another */
 norm[Z]=u[X]*v[Y]-u[Y]*v[X];

 lng=(GLfloat)sqrt(norm[X]*norm[X]+norm[Y]*norm[Y]+norm[Z]*norm[Z]);

 norm[X]/=lng; norm[Y]/=lng; norm[Z]/=lng;

 return(norm);
}

/**********************************************************\
 * Same as above but out of two vectors.                  *
\**********************************************************/

GLfloat* vectorNormalV(GLfloat* norm, GLfloat* u, GLfloat* v)
{
 GLfloat lng;

 norm[X]=u[Y]*v[Z]-u[Z]*v[Y];               /* primitive functions */
 norm[Y]=u[Z]*v[X]-u[X]*v[Z];               /* don't call one another */
 norm[Z]=u[X]*v[Y]-u[Y]*v[X];

 lng=(GLfloat)sqrt(norm[X]*norm[X]+norm[Y]*norm[Y]+norm[Z]*norm[Z]);

 norm[X]/=lng; norm[Y]/=lng; norm[Z]/=lng;

 return(norm);
}

/**********************************************************\
 * Multiplying a matrix and a column                      *
\**********************************************************/

GLfloat* matrixMultiply(GLfloat result[4],GLfloat matrix[4][4],GLfloat column[4])
{
 result[X]=matrix[X][X]*column[X]+matrix[Y][X]*column[Y]+matrix[Z][X]*column[Z]+matrix[W][X]*column[3];
 result[Y]=matrix[X][Y]*column[X]+matrix[Y][Y]*column[Y]+matrix[Z][Y]*column[Z]+matrix[W][Y]*column[3];
 result[Z]=matrix[X][Z]*column[X]+matrix[Y][Z]*column[Y]+matrix[Z][Z]*column[Z]+matrix[W][Z]*column[3];
 result[W]=matrix[X][W]*column[X]+matrix[Y][W]*column[Y]+matrix[Z][W]*column[Z]+matrix[W][W]*column[W];
 
 return(result);
}

/**********************************************************\
 * Compute plane's equation given three points.           *
\**********************************************************/

struct Plane* planeConstruct(struct Plane* plane, GLfloat* a, GLfloat* b, GLfloat* c)
{
 vectorNormal(plane->n,a,b,c);              /* higher level functions */
 vectorCopy(plane->p,a);                    /* use lower level ones */

 return(plane);
}

/**********************************************************\
 * Advance plane by a distance along the normal.          *
\**********************************************************/

struct Plane* planeAdvance(struct Plane* pResPlane, struct Plane* pPlane, GLfloat distance)
{
 GLfloat vdist[4];

 *pResPlane=*pPlane;
 vectorScale(vdist,pResPlane->n,distance);  /* distance as a vector */
 vectorAdd(pResPlane->p,pResPlane->p,vdist);

 return(pResPlane);
}

/**********************************************************\
 * Construct a ray given two points.                      *
\**********************************************************/

struct Ray* rayConstruct(struct Ray* ray, GLfloat* a, GLfloat* b)
{
 vectorConstruct(ray->v,b,a);
 vectorNormalize(ray->v);
 vectorCopy(ray->p,a);

 return(ray);
}

/**********************************************************\
 * Find world point given ray parameter.                  *
\**********************************************************/

GLfloat* rayPointOn(GLfloat* point, struct Ray* ray, GLfloat t)
{
 vectorCopy(point, ray->v);
 vectorScale(point, point, t);
 vectorAdd(point, point, ray->p);

 return(point);
}

/**********************************************************\
 * Compute an intersection of a ray and a plane.          *
 *                                                        *
 * vector:   X=Pv+tC                                      *
 * plane:    (X-Pp)N=0                                    *
 *                                                        *
 * (Pv+tC-Pp)N=0  =>  t(CN)=(Pp-Pv)N  =>                  *
 * => t=((Pp-Pv)N)/(CN)                                   *
\**********************************************************/

GLfloat rayIntersectPlane(struct Ray* ray, struct Plane* plane)
{
 GLfloat df[4];
 GLfloat t;

 vectorSub(df,plane->p,ray->p);
 t=vectorScalarProduct(df,plane->n)/vectorScalarProduct(ray->v,plane->n); 

 return(t);
}

/**********************************************************/
