/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Drawing of shadow maps.                                *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include "dslshmap.h"
#include "dslutil.h"                        /* vector math */
#include "dslofscr.h"                       /* offscreen drawing */
#include "dslworld.h"                       /* scene manager */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Light shadowMapLight=
{
 {1.0, 1.0, 1.0},
 {0.0, 0.0, 0.0},
 {0.0, 0.0, 0.0},
 {0.0, 0.0, 0.0, 1.0},                      /* ambient light */

 0.0f, 0.03f, 0.0f                          /* attenuated with distance */
};

struct Material shadowMapMaterial=
{
 {1.0, 1.0, 1.0},                           /* used with the light above */
 {0.0, 0.0, 0.0},
 {0.0, 0.0, 0.0},
 0
};

GLfloat shadowMapMatrix[4][4];            

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Create a light source view of the scene.               *
\**********************************************************/

void shadowMapDrawLightSourceMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat* mapCenter, GLfloat mapFovy)
{
 GLfloat l[4],p[4],n[4];
 GLfloat j[4]={0.0, 1.0, 0.0, 1.0};
 int i;

 vectorConstruct(l,lightPosition,mapCenter);
 vectorNormalize(l);                        /* z axis */
 vectorNormalV(p,j,l);                      /* x axis */

 if(vectorLength(p)!=0.0)
 {
  vectorNormalV(n,l,p);                     /* y axis */

  shadowMapMatrix[X][X]=p[X]; 
  shadowMapMatrix[Y][X]=p[Y]; 
  shadowMapMatrix[Z][X]=p[Z]; 
  shadowMapMatrix[W][X]=0.0f;

  shadowMapMatrix[X][Y]=n[X]; 
  shadowMapMatrix[Y][Y]=n[Y]; 
  shadowMapMatrix[Z][Y]=n[Z]; 
  shadowMapMatrix[W][Y]=0.0f;

  shadowMapMatrix[X][Z]=l[X]; 
  shadowMapMatrix[Y][Z]=l[Y]; 
  shadowMapMatrix[Z][Z]=l[Z]; 
  shadowMapMatrix[W][Z]=0.0f;

  shadowMapMatrix[X][W]=0.0f; 
  shadowMapMatrix[Y][W]=0.0f; 
  shadowMapMatrix[Z][W]=0.0f; 
  shadowMapMatrix[W][W]=1.0f;
 }
 else                                       /* directly above or under */
 {
  shadowMapMatrix[X][X]=1.0f; 
  shadowMapMatrix[Y][X]=0.0f; 
  shadowMapMatrix[Z][X]=0.0f; 
  shadowMapMatrix[W][X]=0.0f;

  shadowMapMatrix[X][Y]=0.0f; 
  shadowMapMatrix[Y][Y]=0.0f; 
  shadowMapMatrix[Z][Y]=1.0f; 
  shadowMapMatrix[W][Y]=0.0f;

  shadowMapMatrix[X][Z]=l[X]; 
  shadowMapMatrix[Y][Z]=l[Y]; 
  shadowMapMatrix[Z][Z]=l[Z]; 
  shadowMapMatrix[W][Z]=0.0f;

  shadowMapMatrix[X][W]=0.0f; 
  shadowMapMatrix[Y][W]=0.0f; 
  shadowMapMatrix[Z][W]=0.0f; 
  shadowMapMatrix[W][W]=1.0f;
 }

 glEnable(GL_LIGHTING);  
 glDisable(GL_TEXTURE_2D);

 offscreenBegin(OFFSCREEN_BLACK,1);         /* draw model offscreen */
 for(i=0;i<nModelInstances;i++)
 {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  materialEnable(&shadowMapMaterial);       /* with distance */
  lightEnable(&shadowMapLight,0);           /* only ambient, attenuated */
                                            /* light source transform */
  glMultMatrixf((GLfloat*)shadowMapMatrix);
  glTranslatef(-lightPosition[X],-lightPosition[Y],-lightPosition[Z]);
  glTranslatef(pModelInstances[i]->position[X],pModelInstances[i]->position[Y],pModelInstances[i]->position[Z]);
  glRotatef(pModelInstances[i]->orientation[X],1.0f,0.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Y],0.0f,1.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Z],0.0f,0.0f,1.0f);
  glScalef(1.02f,1.02f,1.02f);              /* helps with artefacts */

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();                         /* projection accounted for in modview */
  gluPerspective(mapFovy, 
                 ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                 SHADOW_MAP_NEAR_Z, SHADOW_MAP_FAR_Z);
                                            
  modelInstanceEmit(pModelInstances[i],0,0);/* no normals */

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
 }
 offscreenEnd();
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a camera view of the scene using projective       *
 * texture.                                               *
\**********************************************************/

void shadowMapDrawCameraWithMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat mapFovy)
{
 GLfloat planeS[4]={1.0, 0.0, 0.0, 0.0};
 GLfloat planeT[4]={0.0, 1.0, 0.0, 0.0};
 GLfloat planeR[4]={0.0, 0.0, 1.0, 0.0};
 GLfloat planeQ[4]={0.0, 0.0, 0.0, 1.0};
 int i;

 glEnable(GL_TEXTURE_GEN_S);                /* enable projective texture */
 glEnable(GL_TEXTURE_GEN_T);
 glEnable(GL_TEXTURE_GEN_R);
 glEnable(GL_TEXTURE_GEN_Q);

 glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);

 glTexGenfv(GL_S,GL_OBJECT_PLANE,planeS);
 glTexGenfv(GL_T,GL_OBJECT_PLANE,planeT);
 glTexGenfv(GL_R,GL_OBJECT_PLANE,planeR);
 glTexGenfv(GL_Q,GL_OBJECT_PLANE,planeQ);

 glDisable(GL_LIGHT0);                      /* create a lightsource */
 glDisable(GL_LIGHTING);  
 glEnable(GL_TEXTURE_2D);                   /* shadow is in the texture */
 glColor3f(1.0,1.0,1.0);

 for(i=0;i<nModelInstances;i++)
 {
  glMatrixMode(GL_TEXTURE);
  glPushMatrix();
  glLoadIdentity();                         /* projection accounted for in modview */
  glTranslatef(0.5,0.5,0.0);                /* move to the middle of offscreen */
  glScalef(0.5,0.5,1.0);                    /* texture space -1.0 to 1.0 */
  gluPerspective(mapFovy, 
                 ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                 SHADOW_MAP_NEAR_Z, SHADOW_MAP_FAR_Z);
  glMultMatrixf((GLfloat*)shadowMapMatrix);
  glTranslatef(-lightPosition[X],-lightPosition[Y],-lightPosition[Z]);
  glTranslatef(pModelInstances[i]->position[X],pModelInstances[i]->position[Y],pModelInstances[i]->position[Z]);
  glRotatef(pModelInstances[i]->orientation[X],1.0f,0.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Y],0.0f,1.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Z],0.0f,0.0f,1.0f);

  modelInstanceEmit(pModelInstances[i],1,0);/* does modelview transform */

  glMatrixMode(GL_TEXTURE);
  glPopMatrix();
 }
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a camera view of the scene.                       *
\**********************************************************/

void shadowMapDrawCamera(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition)
{
 int i;

 shadowMapLight.position[X]=lightPosition[X];
 shadowMapLight.position[Y]=lightPosition[Y];
 shadowMapLight.position[Z]=lightPosition[Z];
 shadowMapLight.position[W]=lightPosition[W];
                                            /* world position of the light */
 materialEnable(&shadowMapMaterial);        /* with distance */
 lightEnable(&shadowMapLight,0);            /* only ambient, attenuated */

 glEnable(GL_LIGHTING);  
 glDisable(GL_TEXTURE_2D);                  /* shadow is in the texture */

 for(i=0;i<nModelInstances;i++)
  modelInstanceEmit(pModelInstances[i],1,0);

 shadowMapLight.position[X]=0.0f;           /* back to the lightsource in the */
 shadowMapLight.position[Y]=0.0f;           /* eye */
 shadowMapLight.position[Z]=0.0f;
 shadowMapLight.position[W]=1.0f;
}

/**********************************************************\
 * Create a shadow map. Involves drawing into a           *
 * projective texture.                                    *
\**********************************************************/

void shadowMapDrawMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat* mapCenter, GLfloat mapFovy)
{
 struct Light* pActualLight=pCurrentLight;  /* remember lighting */
 int nActualLight=nCurrentLight;

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 shadowMapDrawLightSourceMap(pModelInstances,nModelInstances,lightPosition,mapCenter,mapFovy);

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 shadowMapDrawCameraWithMap(pModelInstances,nModelInstances,lightPosition,mapFovy);

 glReadBuffer(GL_BACK);                     /* distances from the light */
 glAccum(GL_LOAD,1.0);                      /* source accomulated */

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 shadowMapDrawCamera(pModelInstances,nModelInstances,lightPosition);

 glReadBuffer(GL_BACK);
 glAccum(GL_ACCUM,-1.0f);

 glAccum(GL_ADD,-0.02f);                    /* get rid of some artefacts */

 glDrawBuffer(GL_BACK);
 glAccum(GL_RETURN,16.0f);                  /* scale up shadows (with clamping) */
 glAccum(GL_LOAD,-1.0f);                    /* read beck */

 lightEnable(pActualLight,nActualLight);    /* reenable lighting */
}

/**********************************************************\
 * Draw the scene and blend over the shadow map.          *
\**********************************************************/

void shadowMapDrawScene(struct ModelInstance** pModelInstances, int nModelInstances)
{
 int i;

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 for(i=0;i<nModelInstances;i++)             /* draw all models */
  modelInstanceDrawSolid(pModelInstances[i]);

 glAccum(GL_ACCUM,1.0);                     /* blend subtractively with */
 glAccum(GL_RETURN,1.0);                    /* the shadow map */
}

/**********************************************************/
