/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Drawing of projective textures.                        *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include "dslproj.h"          
#include "dslutil.h"                        /* vector math */
#include "dslofscr.h"                       /* offscreen drawing */

GLfloat projectiveTextureMatrix[4][4];      /* set by .Cater used by .Castee */
GLfloat projectiveTextureFovy;              /* field of view for y */

/**********************************************************\
 * Draw a model from the position of a light source into  *
 * an offscreen buffer.                                   *
 *                                                        *
 *                                                        *
 * if L is the position of the lightsource and O position *
 * of the object. The axes for the viewing coordinate     *
 * ssystem could be obtained as:                          *
 *                                                        *
 *    L-O          (0,1,0)xl        lxp                   *
 * l=-------   p=-------------  n=-------                 *
 *   ||L-O||     ||(0,1,0)xl||    ||lxp||                 *
 *                                                        *
 * and thus the viewing transform from model into         *
 * lightsource view is:        -1          T              *
 *                      [p n l]   = [p n l]               *
 * rotations only matrices are orthonormal their inverse  *
 * is their transposition.                                *
 *                                                        *
 * if looking directly down and (0,1,0)xl is 0 build      *
 * transform by hand.                                     *
 *                                                        *
\**********************************************************/

void projectiveTextureDrawCaster(struct ModelInstance* pModelInstance, GLfloat* lightPosition)
{
 GLfloat l[4],p[4],n[4];
 GLfloat j[4]={0.0, 1.0, 0.0, 1.0};
 GLfloat lng; 

 vectorConstruct(l,lightPosition,pModelInstance->position);
 lng=vectorLength(l);
 
 if(lng==0.0) return;                       /* model at the lightsource */

 vectorNormalize(l);                        /* z axis */
 vectorNormalV(p,j,l);                      /* x axis */

 if(vectorLength(p)!=0.0)
 {
  vectorNormalV(n,l,p);                     /* y axis */

  projectiveTextureMatrix[X][X]=p[X]; 
  projectiveTextureMatrix[Y][X]=p[Y]; 
  projectiveTextureMatrix[Z][X]=p[Z]; 
  projectiveTextureMatrix[W][X]=0.0f;

  projectiveTextureMatrix[X][Y]=n[X]; 
  projectiveTextureMatrix[Y][Y]=n[Y]; 
  projectiveTextureMatrix[Z][Y]=n[Z]; 
  projectiveTextureMatrix[W][Y]=0.0f;

  projectiveTextureMatrix[X][Z]=l[X]; 
  projectiveTextureMatrix[Y][Z]=l[Y]; 
  projectiveTextureMatrix[Z][Z]=l[Z]; 
  projectiveTextureMatrix[W][Z]=0.0f;

  projectiveTextureMatrix[X][W]=0.0f; 
  projectiveTextureMatrix[Y][W]=0.0f; 
  projectiveTextureMatrix[Z][W]=0.0f; 
  projectiveTextureMatrix[W][W]=1.0f;
 }
 else                                       /* directly above or under */
 {
  projectiveTextureMatrix[X][X]=1.0f; 
  projectiveTextureMatrix[Y][X]=0.0f; 
  projectiveTextureMatrix[Z][X]=0.0f; 
  projectiveTextureMatrix[W][X]=0.0f;

  projectiveTextureMatrix[X][Y]=0.0f; 
  projectiveTextureMatrix[Y][Y]=0.0f; 
  projectiveTextureMatrix[Z][Y]=1.0f; 
  projectiveTextureMatrix[W][Y]=0.0f;

  projectiveTextureMatrix[X][Z]=l[X]; 
  projectiveTextureMatrix[Y][Z]=l[Y]; 
  projectiveTextureMatrix[Z][Z]=l[Z]; 
  projectiveTextureMatrix[W][Z]=0.0f;

  projectiveTextureMatrix[X][W]=0.0f; 
  projectiveTextureMatrix[Y][W]=0.0f; 
  projectiveTextureMatrix[Z][W]=0.0f; 
  projectiveTextureMatrix[W][W]=1.0f;
 }
                                            /* *2 since atan gives half angle */
 projectiveTextureFovy=(GLfloat)((atan(pModelInstance->pModel->boundingRadius/lng)*360.0)/3.14);

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glLoadIdentity();

 glTranslatef(0.0,0.0,-lng);
                                            /* light source transform */
 glMultMatrixf((GLfloat*)projectiveTextureMatrix);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glMatrixMode(GL_PROJECTION);
 glPushMatrix();
 glLoadIdentity();                          /* projection accounted for in modview */
 gluPerspective(projectiveTextureFovy, 
                ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                PROJECTIVE_TEXTURE_NEAR_Z, PROJECTIVE_TEXTURE_FAR_Z);

 offscreenBegin(OFFSCREEN_WHITE,1);         /* draw model offscreen */
 glDisable(GL_LIGHTING);
 glDisable(GL_TEXTURE_2D);
 glColor3f(0.0, 0.0, 0.0);
 modelInstanceEmit(pModelInstance,0,0);     /* no normals */
 offscreenEnd();

 glMatrixMode(GL_PROJECTION);
 glPopMatrix();
 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 modelInstanceDrawSolid(pModelInstance);
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a model receiving shadow.                         *
\**********************************************************/

void projectiveTextureDrawReceiver(struct ModelInstance* pModelInstance, GLfloat* lightPosition)
{
 GLfloat planeS[4]={1.0, 0.0, 0.0, 0.0};
 GLfloat planeT[4]={0.0, 1.0, 0.0, 0.0};
 GLfloat planeR[4]={0.0, 0.0, 1.0, 0.0};
 GLfloat planeQ[4]={0.0, 0.0, 0.0, 1.0};

 glEnable(GL_TEXTURE_GEN_S);
 glEnable(GL_TEXTURE_GEN_T);
 glEnable(GL_TEXTURE_GEN_R);
 glEnable(GL_TEXTURE_GEN_Q);

 glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);

 glTexGenfv(GL_S,GL_OBJECT_PLANE,planeS);
 glTexGenfv(GL_T,GL_OBJECT_PLANE,planeT);
 glTexGenfv(GL_R,GL_OBJECT_PLANE,planeR);
 glTexGenfv(GL_Q,GL_OBJECT_PLANE,planeQ);

 glMatrixMode(GL_TEXTURE);
 glPushMatrix();
 glLoadIdentity();                          /* projection accounted for in modview */
 glTranslatef(0.5,0.5,0.0);                 /* move to the middle of offscreen */
 glScalef(0.5,0.5,1.0);                     /* texture space -1.0 to 1.0 */

 gluPerspective(projectiveTextureFovy, 
                ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                PROJECTIVE_TEXTURE_NEAR_Z, PROJECTIVE_TEXTURE_FAR_Z);
 glMultMatrixf((GLfloat*)projectiveTextureMatrix);
 glTranslatef(-lightPosition[X],-lightPosition[Y],-lightPosition[Z]);
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glEnable(GL_LIGHTING);  
 glEnable(GL_TEXTURE_2D);                   /* shadow is in the texture */
 modelInstanceDrawSolid(pModelInstance);    /* does modelview transform */

 glDisable(GL_TEXTURE_2D);                  /* no textures on models */
 glMatrixMode(GL_TEXTURE);
 glPopMatrix();
}

/**********************************************************\
 * Draw all receivers texturing them with the image of    *
 * the shadow caster.                                     *
\**********************************************************/

void projectiveTextureDrawScene(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition)
{
 int i;

 for(i=0;i<nModelInstances;i++)
  projectiveTextureDrawReceiver(pModelInstances[i],lightPosition);
}

/**********************************************************/
