/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Drawing of priority maps.                              *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include "dslprmap.h"
#include "dslutil.h"                        /* vector math */
#include "dslofscr.h"                       /* offscreen drawing */
#include "dslworld.h"                       /* scene manager */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

GLfloat priorityMapMatrix[4][4];            
GLfloat priorityMapColors[MAX_MODEL_INSTANCES];

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Create a light source view of the scene.               *
\**********************************************************/

void priorityMapDrawLightSourceMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat* mapCenter, GLfloat mapFovy)
{
 GLfloat l[4],p[4],n[4];
 GLfloat j[4]={0.0, 1.0, 0.0, 1.0};
 int i,k;
 GLfloat lngs[MAX_MODEL_INSTANCES];
 int idx[MAX_MODEL_INSTANCES];

 for(i=0;i<nModelInstances;i++)             /* compute distances to light source */
 {
  lngs[i]=vectorLength(vectorConstruct(l,pModelInstances[i]->position,lightPosition));
  idx[i]=i;
 }

 for(i=1;i<nModelInstances;i++)             /* sort distances */
  for(k=0;k<nModelInstances-i;k++)
  {
   if(lngs[k]>lngs[k+1])
   {
    GLfloat tmpf;
    int tmpi;

    tmpf=lngs[k+1]; lngs[k+1]=lngs[k]; lngs[k]=tmpf;
    tmpi=idx[k+1]; idx[k+1]=idx[k]; idx[k]=tmpi;
   }
  }

 for(i=0;i<nModelInstances;i++)             /* assign colors based on distance */
  priorityMapColors[idx[i]]=(GLfloat)(1.0-(i)*(1.0/(nModelInstances)));

 vectorConstruct(l,lightPosition,mapCenter);
 vectorNormalize(l);                        /* z axis */
 vectorNormalV(p,j,l);                      /* x axis */

 if(vectorLength(p)!=0.0)
 {
  vectorNormalV(n,l,p);                     /* y axis */

  priorityMapMatrix[X][X]=p[X]; 
  priorityMapMatrix[Y][X]=p[Y]; 
  priorityMapMatrix[Z][X]=p[Z]; 
  priorityMapMatrix[W][X]=0.0f;

  priorityMapMatrix[X][Y]=n[X]; 
  priorityMapMatrix[Y][Y]=n[Y]; 
  priorityMapMatrix[Z][Y]=n[Z]; 
  priorityMapMatrix[W][Y]=0.0f;

  priorityMapMatrix[X][Z]=l[X]; 
  priorityMapMatrix[Y][Z]=l[Y]; 
  priorityMapMatrix[Z][Z]=l[Z]; 
  priorityMapMatrix[W][Z]=0.0f;

  priorityMapMatrix[X][W]=0.0f; 
  priorityMapMatrix[Y][W]=0.0f; 
  priorityMapMatrix[Z][W]=0.0f; 
  priorityMapMatrix[W][W]=1.0f;
 }
 else                                       /* directly above or under */
 {
  priorityMapMatrix[X][X]=1.0f; 
  priorityMapMatrix[Y][X]=0.0f; 
  priorityMapMatrix[Z][X]=0.0f; 
  priorityMapMatrix[W][X]=0.0f;

  priorityMapMatrix[X][Y]=0.0f; 
  priorityMapMatrix[Y][Y]=0.0f; 
  priorityMapMatrix[Z][Y]=1.0f; 
  priorityMapMatrix[W][Y]=0.0f;

  priorityMapMatrix[X][Z]=l[X]; 
  priorityMapMatrix[Y][Z]=l[Y]; 
  priorityMapMatrix[Z][Z]=l[Z]; 
  priorityMapMatrix[W][Z]=0.0f;

  priorityMapMatrix[X][W]=0.0f; 
  priorityMapMatrix[Y][W]=0.0f; 
  priorityMapMatrix[Z][W]=0.0f; 
  priorityMapMatrix[W][W]=1.0f;
 }

 glDisable(GL_LIGHTING);  
 glDisable(GL_TEXTURE_2D);

 offscreenBegin(OFFSCREEN_BLACK,1);         /* draw model offscreen */
 for(i=0;i<nModelInstances;i++)
 {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glColor3f(priorityMapColors[i],priorityMapColors[i],priorityMapColors[i]);
                                            /* light source transform */
  glMultMatrixf((GLfloat*)priorityMapMatrix);
  glTranslatef(-lightPosition[X],-lightPosition[Y],-lightPosition[Z]);
  glTranslatef(pModelInstances[i]->position[X],pModelInstances[i]->position[Y],pModelInstances[i]->position[Z]);
  glRotatef(pModelInstances[i]->orientation[X],1.0f,0.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Y],0.0f,1.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Z],0.0f,0.0f,1.0f);
  glScalef(1.02f,1.02f,1.02f);              /* helps with artefacts */

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();                         /* projection accounted for in modview */
  gluPerspective(mapFovy, 
                 ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                 PRIORITY_MAP_NEAR_Z, PRIORITY_MAP_FAR_Z);
                                            
  modelInstanceEmit(pModelInstances[i],0,0);/* no normals */

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
 }
 offscreenEnd();
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a camera view of the scene using projective       *
 * texture.                                               *
\**********************************************************/

void priorityMapDrawCameraWithMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat mapFovy)
{
 GLfloat planeS[4]={1.0, 0.0, 0.0, 0.0};
 GLfloat planeT[4]={0.0, 1.0, 0.0, 0.0};
 GLfloat planeR[4]={0.0, 0.0, 1.0, 0.0};
 GLfloat planeQ[4]={0.0, 0.0, 0.0, 1.0};
 int i;

 glEnable(GL_TEXTURE_GEN_S);                /* enable projective texture */
 glEnable(GL_TEXTURE_GEN_T);
 glEnable(GL_TEXTURE_GEN_R);
 glEnable(GL_TEXTURE_GEN_Q);

 glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
 glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);

 glTexGenfv(GL_S,GL_OBJECT_PLANE,planeS);
 glTexGenfv(GL_T,GL_OBJECT_PLANE,planeT);
 glTexGenfv(GL_R,GL_OBJECT_PLANE,planeR);
 glTexGenfv(GL_Q,GL_OBJECT_PLANE,planeQ);

 glDisable(GL_LIGHT0);                      /* create a lightsource */
 glDisable(GL_LIGHTING);  
 glEnable(GL_TEXTURE_2D);                   /* shadow is in the texture */
 glColor3f(1.0,1.0,1.0);

 for(i=0;i<nModelInstances;i++)
 {
  glMatrixMode(GL_TEXTURE);
  glPushMatrix();
  glLoadIdentity();                         /* projection accounted for in modview */
  glTranslatef(0.5,0.5,0.0);                /* move to the middle of offscreen */
  glScalef(0.5,0.5,1.0);                    /* texture space -1.0 to 1.0 */
  gluPerspective(mapFovy, 
                 ((GLfloat)offscreenSizeX)/offscreenSizeY, 
                 PRIORITY_MAP_NEAR_Z, PRIORITY_MAP_FAR_Z);
  glMultMatrixf((GLfloat*)priorityMapMatrix);
  glTranslatef(-lightPosition[X],-lightPosition[Y],-lightPosition[Z]);
  glTranslatef(pModelInstances[i]->position[X],pModelInstances[i]->position[Y],pModelInstances[i]->position[Z]);
  glRotatef(pModelInstances[i]->orientation[X],1.0f,0.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Y],0.0f,1.0f,0.0f);
  glRotatef(pModelInstances[i]->orientation[Z],0.0f,0.0f,1.0f);

  modelInstanceEmit(pModelInstances[i],1,0);/* does modelview transform */

  glMatrixMode(GL_TEXTURE);
  glPopMatrix();
 }
}

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a camera view of the scene.                       *
\**********************************************************/

void priorityMapDrawCamera(struct ModelInstance** pModelInstances, int nModelInstances)
{
 int i;

 glDisable(GL_LIGHTING);  
 glDisable(GL_TEXTURE_2D);                  /* shadow is in the texture */

 for(i=0;i<nModelInstances;i++)
 {
  glColor3f(priorityMapColors[i],priorityMapColors[i],priorityMapColors[i]);
  modelInstanceEmit(pModelInstances[i],1,0);
 }
}

/**********************************************************\
 * Create a shadow map. Involves drawing into a           *
 * projective texture.                                    *
\**********************************************************/

void priorityMapDrawMap(struct ModelInstance** pModelInstances, int nModelInstances, GLfloat* lightPosition, GLfloat* mapCenter, GLfloat mapFovy)
{
 struct Light* pActualLight=pCurrentLight;  /* remember lighting */
 int nActualLight=nCurrentLight;

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 priorityMapDrawLightSourceMap(pModelInstances,nModelInstances,lightPosition,mapCenter,mapFovy);

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 priorityMapDrawCameraWithMap(pModelInstances,nModelInstances,lightPosition,mapFovy);

 glReadBuffer(GL_BACK);                     /* distances from the light */
 glAccum(GL_LOAD,1.0);                      /* source accomulated */

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 priorityMapDrawCamera(pModelInstances,nModelInstances);

 glReadBuffer(GL_BACK);
 glAccum(GL_ACCUM,-1.0f);

 glAccum(GL_ADD,-0.02f);                    /* get rid of some artefacts */

 glDrawBuffer(GL_BACK);
 glAccum(GL_RETURN,16.0f);                  /* scale up shadows (with clamping) */
 glAccum(GL_LOAD,-1.0f);                    /* read beck */

 lightEnable(pActualLight,nActualLight);    /* reenable lighting */
}

/**********************************************************\
 * Draw the scene and blend over the shadow map.          *
\**********************************************************/

void priorityMapDrawScene(struct ModelInstance** pModelInstances, int nModelInstances)
{
 int i;

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 for(i=0;i<nModelInstances;i++)             /* draw all models */
  modelInstanceDrawSolid(pModelInstances[i]);

 glAccum(GL_ACCUM,1.0);                     /* blend subtractively with */
 glAccum(GL_RETURN,1.0);                    /* the shadow map */
}

/**********************************************************/
