/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Drawing of planar shadows.                             *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include "dslplanr.h"

/**********************************************************\
 * Internal.                                              *
 *                                                        *
 * Draw a model projected onto a plane. Stencil buffer    *
 * is tested to see if the shadow pixel is inside the     *
 * planar surface (which were marked in the stencil       *
 * buffer).                                               *
 *                                                        *
 * For lightsource l, plane nx-np=0 (point x on the plane)*
 * and m point in space:                                  *
 *                                                        *
 * (x-l)   (nl-np)          (nl-np)                       *
 * ----- = -------  =>  x=l+-------(m-l) =>               *
 * (m-l)   n(l-m)           (nl-nm)                       *
 *                                                        *
 *                                                        *
 *    l(nl-nm)+m(nl-np)-l(nl-np)  l(nl-nm-nl+np)+m(nl-np) *
 * x= --------------------------= ----------------------- *
 *            nl-nm                       nl-nm           *
 *                                                        *
 *    l(np-nm)+m(nl-np)   lnp - lnm + m(nl-np)            *
 * x= ----------------- = --------------------            *
 *         nl-nm                nl-nm                     *
 *                                                        *
 *     |l_xnp-l_x(n_x*m_x+n_y*m_y+n_z*m_z) + m_x(nl-np)|  *
 *     |-----------------------------------------------|  *
 *     |        nl - n_x*m_x - n_y*m_y - n_z*m_z       |  *
 * |x| |l_ynp-l_y(n_x*m_x+n_y*m_y+n_z*m_z) + m_y(nl-np)|  *
 * |y|=|-----------------------------------------------|  *
 * |z| |        nl - n_x*m_x - n_y*m_y - n_z*m_z       |  *
 *     |l_znp-l_z(n_x*m_x+n_y*m_y+n_z*m_z) + m_z(nl-np)|  *
 *     |-----------------------------------------------|  *
 *     |        nl - n_x*m_x - n_y*m_y - n_z*m_z       |  *
 *                                                        *
 * or in matrix form x=Mm M is:                           *
 *                                                        *
 * |nl-np-l_x*n_x     -l_x*n_y      -l_x*n_z      l_xnp|  *
 * |   -l_y*n_x     nl-np-l_y*n_y   -l_y*n_z      l_ynp|  *                                         
 * |   -l_z*n_x       -l_z*n_y     nl-np-l_z*n_z  l_znp|  *             
 * |    -n_x            -n_y          -n_z          nl |  *
 *                                                        *
\**********************************************************/

void planarShadowDrawCaster(struct ModelInstance* pModelInstance, struct Plane* pPlane, GLfloat* lightposition)
{
 GLfloat shadow[4][4];
 GLfloat nl,np;
 struct Plane advPlane;

 glEnable(GL_STENCIL_TEST);
 glStencilFunc(GL_EQUAL,1,1);
 glStencilOp(GL_KEEP,GL_KEEP,GL_KEEP);
                                            /* to avoid Z fighting */
 planeAdvance(&advPlane,pPlane,PLANAR_ZFIGHT_DIST);

 nl=vectorScalarProduct(pPlane->n,lightposition);
 np=vectorScalarProduct(pPlane->n,advPlane.p);

 shadow[X][X]=nl-np-lightposition[X]*pPlane->n[X];
 shadow[Y][X]=-lightposition[X]*pPlane->n[Y];
 shadow[Z][X]=-lightposition[X]*pPlane->n[Z];
 shadow[W][X]=lightposition[X]*np;

 shadow[X][Y]=-lightposition[Y]*pPlane->n[X];
 shadow[Y][Y]=nl-np-lightposition[Y]*pPlane->n[Y];
 shadow[Z][Y]=-lightposition[Y]*pPlane->n[Z];
 shadow[W][Y]=lightposition[Y]*np;

 shadow[X][Z]=-lightposition[Z]*pPlane->n[X];
 shadow[Y][Z]=-lightposition[Z]*pPlane->n[Y];
 shadow[Z][Z]=nl-np-lightposition[Z]*pPlane->n[Z];
 shadow[W][Z]=lightposition[Z]*np;

 shadow[X][W]=-pPlane->n[X];
 shadow[Y][W]=-pPlane->n[Y];
 shadow[Z][W]=-pPlane->n[Z];
 shadow[W][W]=nl;

 glDisable(GL_LIGHTING);
 glColor3f(0.0, 0.0, 0.0);                  /* practically black */
 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glMultMatrixf((GLfloat*)shadow);           /* squish transform */
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 modelInstanceEmit(pModelInstance,0,0);     /* don't emit normals */

 glPopMatrix();

 glDisable(GL_STENCIL_TEST);
}

/**********************************************************\
 * Draw the plane receiving shadow.                       *
\**********************************************************/

void planarShadowDrawReceiver(struct ModelInstance* pModelInstance)
{
 glClearStencil(0);
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
 glEnable(GL_STENCIL_TEST);

 glStencilFunc(GL_ALWAYS,1,1);
 glStencilOp(GL_REPLACE,GL_REPLACE,GL_REPLACE);

 modelInstanceDrawSolid(pModelInstance);

 glDisable(GL_STENCIL_TEST);
}

/**********************************************************\
 * Draw all casters casting onto the same plane.          *
\**********************************************************/

void planarShadowDrawScene(struct ModelInstance** pModelInstances, int nModelInstances, struct Plane* pPlane, GLfloat* lightPosition)
{
 int i;

 for(i=0;i<nModelInstances;i++)
 {
  planarShadowDrawCaster(pModelInstances[i],pPlane,lightPosition);
  modelInstanceDrawSolid(pModelInstances[i]);
 }
}

/**********************************************************/
