#ifndef _SDSL_MODEL_H_
#define _SDSL_MODEL_H_

/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Representation of a model.                             *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include <windows.h>                        /* gl.h needs it */
#include <gl/gl.h>                          /* GLfloat etc. */
#include <stdio.h>                          /* FILE */
#include "dslworld.h"                       /* Material */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * A vertex -- contains world coordinates and a normal.   * 
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Vertex 
{ 
 GLfloat v[4];                              /* world coordinates */
 GLfloat n[4];                              /* normal */
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * A triangle -- contains pointers to its vertices and    *
 * triangles normal.                                      * 
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Triangle 
{ 
 struct Vertex *pv[3];                      /* vertex pointers */
 GLfloat n[4];                              /* plane normal */
 GLfloat m[4];                              /* triangles middle */
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * A vertex -- contains world coordinates and a normal.   * 
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Edge
{
 struct Vertex* pv[2];                      /* 2 vertices in an edge */
 struct Triangle* cwt;                      /* clockwise and counter- */
 struct Triangle* ccwt;                     /* clockwise triangles */
};              

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * A vertex -- contains world coordinates and a normal.   * 
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Model
{
 struct Vertex* pVertices;                  /* array of vertices */
 struct Triangle* pTriangles;               /* array of triangles */
 struct Edge* pEdges;                       /* array of edges */

 int nVertices;
 int nTriangles;
 int nEdges;

 struct Material* pMaterial;                /* entire model of one material */
                                            /* hmmm... oh well.. */
 GLfloat boundingRadius;                    /* radius of the binding sphere */
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct Model* modelCreate(FILE* f);
void modelDestroy(struct Model* pModel);

void modelScale(struct Model* pModel, GLfloat scl);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * A model placed in the world.                           *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

#define MAX_MODEL_INSTANCES 16              /* bound for some static arrays */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

struct ModelInstance
{
 struct Model* pModel;

 GLfloat position[4];                       /* in world space */
 GLfloat orientation[3];                    /* three angles */
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

void modelInstanceCreate(struct ModelInstance* pModelInstance, struct Model* pModel);
void modelInstanceSetPosition(struct ModelInstance* pModelInstance, GLfloat x, GLfloat y, GLfloat z);
void modelInstanceSetOrientation(struct ModelInstance* pModelInstance,GLfloat xa, GLfloat ya, GLfloat za); 
void modelInstanceAddPosition(struct ModelInstance* pModelInstance, int axis, GLfloat val);
void modelInstanceAddOrientation(struct ModelInstance* pModelInstance, int axis, GLfloat val);

void modelInstanceEmit(struct ModelInstance* pModelInstance, int emitTransforms, int emitNormals);
void modelInstanceDrawSolid(struct ModelInstance* pModelInstance);
void modelInstanceDrawWire(struct ModelInstance* pModelInstance);

void modelInstanceDrawEdges(struct ModelInstance* pModelInstance);
void modelInstanceDrawTriangleNormals(struct ModelInstance* pModelInstance, GLfloat scale);
void modelInstanceDrawVertexNormals(struct ModelInstance* pModelInstance, GLfloat scale);

/**********************************************************/

#endif
