/**********************************************************\
 * (SDSL2) Small Dynamic Shadows Library.                 *
 *                                                        *
 * Representation of a model.                             *
 *                                                        *
 * Sergei Savchenko 2001.                                 *
\**********************************************************/

#include <string.h>                         /* comparing file tokens */
#include "dslmodel.h"
#include "dslutil.h"                        /* scale etc. */

/**********************************************************\
 * Read a model from an external file.                    *
 * Compute triangle normals, vertex normals and           *
 * edge list.                                             *
\**********************************************************/

struct Model* modelCreate(FILE* f)
{                                           
 char sTag[64];
 int i,j,k;
 struct Edge* pTmpEdges;                    /* unknown due to potential */
 int nTmpEdges;                             /* problems -hanging edges..*/
 struct Model* pModel;

 pModel=(struct Model*)malloc(sizeof(struct Model));
 if(pModel==NULL) return(NULL);

 pModel->pMaterial=materialCreate(f);       /* material goes first in the file */

 fscanf(f,"%s %d",sTag,&pModel->nVertices);
 if(strcmp(sTag,"VERTICES")!=0) return(NULL);
 pModel->pVertices=(struct Vertex*)malloc(pModel->nVertices*sizeof(struct Vertex));
 for(i=0;i<pModel->nVertices;i++)           /* read vertices */
 {
  fscanf(f,"%f %f %f",&pModel->pVertices[i].v[X],&pModel->pVertices[i].v[Y],&pModel->pVertices[i].v[Z]);
 }

 fscanf(f,"%s %d",sTag,&pModel->nTriangles);
 if(strcmp(sTag,"TRIANGLES")!=0) return(NULL);
 pModel->pTriangles=(struct Triangle*)malloc(pModel->nTriangles*sizeof(struct Triangle));
 for(i=0;i<pModel->nTriangles;i++)          /* read triangles */
 {
  int i1,i2,i3;

  fscanf(f,"%d %d %d",&i1,&i2,&i3);
                                            /* replace indices with ptrs */
  pModel->pTriangles[i].pv[0]=&pModel->pVertices[i1];
  pModel->pTriangles[i].pv[1]=&pModel->pVertices[i2];
  pModel->pTriangles[i].pv[2]=&pModel->pVertices[i3];

  vectorNormal(pModel->pTriangles[i].n,     /* compute triangle's normal */
               pModel->pTriangles[i].pv[0]->v,
               pModel->pTriangles[i].pv[1]->v,
               pModel->pTriangles[i].pv[2]->v);

  vectorAdd(pModel->pTriangles[i].m,pModel->pTriangles[i].pv[0]->v,pModel->pTriangles[i].pv[1]->v);
  vectorAdd(pModel->pTriangles[i].m,pModel->pTriangles[i].m,pModel->pTriangles[i].pv[2]->v);
  vectorScale(pModel->pTriangles[i].m,pModel->pTriangles[i].m,1.0f/3.0f);
 }

 for(i=0;i<pModel->nVertices;i++)           /* compute vertex normal */
 {
  int count=0;

  vectorZero(pModel->pVertices[i].n);

  for(j=0;j<pModel->nTriangles;j++)         /* sum over all tri normals */
  {                                         /* containing this vertex */
   if((&pModel->pVertices[i]==pModel->pTriangles[j].pv[0])||
      (&pModel->pVertices[i]==pModel->pTriangles[j].pv[1])||
      (&pModel->pVertices[i]==pModel->pTriangles[j].pv[2]))
   {
    vectorAdd(pModel->pVertices[i].n,pModel->pVertices[i].n,pModel->pTriangles[j].n);

    count++;
   }
  }                                         /* average */

  vectorScale(pModel->pVertices[i].n,pModel->pVertices[i].n,1.0f/count);

  vectorNormalize(pModel->pVertices[i].n);  /* renormalize */
 }

 pTmpEdges=(struct Edge*)malloc(pModel->nTriangles*3*sizeof(struct Edge));
 nTmpEdges=0;
                                            /* count number of edges */
 for(i=0;i<pModel->nTriangles;i++)          /* find edge lists */
 {
  for(j=0;j<3;j++)                          /* tri has 3 edges */
  { 
   for(k=0;k<nTmpEdges;k++)
   {
    if((pModel->pTriangles[i].pv[j]==pTmpEdges[k].pv[1])&&
       (pModel->pTriangles[i].pv[(j+1)%3]==pTmpEdges[k].pv[0]))
    {
     pTmpEdges[k].ccwt=&pModel->pTriangles[i];
     break;
    }
   }
   if(k==nTmpEdges)
   {
    pTmpEdges[nTmpEdges].pv[0]=pModel->pTriangles[i].pv[j];
    pTmpEdges[nTmpEdges].pv[1]=pModel->pTriangles[i].pv[(j+1)%3];
    pTmpEdges[nTmpEdges].cwt=&pModel->pTriangles[i];
    pTmpEdges[nTmpEdges++].ccwt=NULL;
   }
  }
 }
                                            /* replicate tmp structure */
 pModel->pEdges=(struct Edge*)malloc(nTmpEdges*sizeof(struct Edge));
 for(i=0;i<nTmpEdges;i++) pModel->pEdges[i]=pTmpEdges[i];
 pModel->nEdges=nTmpEdges;
 free(pTmpEdges);

 pModel->boundingRadius=0.0f;
 for(i=0;i<pModel->nVertices;i++)
 {
  GLfloat ln=vectorLength(pModel->pVertices[i].v);

  if(ln>pModel->boundingRadius) pModel->boundingRadius=ln;
 }

 return(pModel);
}

/**********************************************************\
 * Deallocate a model.                                    *
\**********************************************************/

void modelDestroy(struct Model* pModel)
{
 materialDestroy(pModel->pMaterial);
 free(pModel->pVertices);
 free(pModel->pTriangles);
 free(pModel->pEdges);
 free(pModel);
}

/**********************************************************\
 * Perform uniform scaling of model's vertices.           *
\**********************************************************/

void modelScale(struct Model* pModel,GLfloat scl)
{
 int i;

 for(i=0;i<pModel->nVertices;i++)
  vectorScale(pModel->pVertices[i].v,pModel->pVertices[i].v,scl);

 for(i=0;i<pModel->nTriangles;i++)
  vectorScale(pModel->pTriangles[i].m,pModel->pTriangles[i].m,scl);

 pModel->boundingRadius*=scl;
}

/**********************************************************\
 * Associate a model instance with a model.               *
\**********************************************************/

void modelInstanceCreate(struct ModelInstance* pModelInstance, struct Model* pModel)
{
 pModelInstance->pModel=pModel;
}

/**********************************************************\
 * Set model's position.                                  *
\**********************************************************/

void modelInstanceSetPosition(struct ModelInstance* pModelInstance, GLfloat x, GLfloat y, GLfloat z)
{
 pModelInstance->position[X]=x; pModelInstance->position[Y]=y; pModelInstance->position[Z]=z;
}

/**********************************************************\
 * Set model's orientation.                               *
\**********************************************************/

void modelInstanceSetOrientation(struct ModelInstance* pModelInstance, GLfloat xa, GLfloat ya, GLfloat za)
{
 pModelInstance->orientation[X]=xa; pModelInstance->orientation[Y]=ya; pModelInstance->orientation[Z]=za;
}

/**********************************************************\
 * Change model's position.                               *
\**********************************************************/

void modelInstanceAddPosition(struct ModelInstance* pModelInstance, int axis, GLfloat val)
{
 pModelInstance->position[axis]+=val;
}

/**********************************************************\
 * Change model's orientation.                            *
\**********************************************************/

void modelInstanceAddOrientation(struct ModelInstance* pModelInstance, int axis, GLfloat val)
{
 pModelInstance->orientation[axis]+=val;
}

/**********************************************************\
 * Emit all triangles of a model without setting          *
 * transforms.                                            *
\**********************************************************/

void modelInstanceEmit(struct ModelInstance* pModelInstance, int emitTransforms, int emitNormals)
{
 int i;
 struct Model* pModel=pModelInstance->pModel;

 if(emitTransforms)
 {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
  glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
  glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
  glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);
 }

 glBegin(GL_TRIANGLES);    
 for(i=0;i<pModel->nTriangles;i++)          /* pass plane or vertex */
 {                                          /* normals */
  if(emitNormals) glNormal3fv(pModel->pTriangles[i].pv[0]->n);
  glVertex3fv(pModel->pTriangles[i].pv[0]->v);

  if(emitNormals) glNormal3fv(pModel->pTriangles[i].pv[1]->n);
  glVertex3fv(pModel->pTriangles[i].pv[1]->v);

  if(emitNormals) glNormal3fv(pModel->pTriangles[i].pv[2]->n);
  glVertex3fv(pModel->pTriangles[i].pv[2]->v);
 }
 glEnd();

 if(emitTransforms)
 {
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
 }
}

/**********************************************************\
 * Draw solid triangles of the model.                     *
 * Emit face normals or vertex normals based on           *
 * isFaceNormal.                                          *
\**********************************************************/

void modelInstanceDrawSolid(struct ModelInstance* pModelInstance)
{
 materialEnable(pModelInstance->pModel->pMaterial);
 glEnable(GL_LIGHTING);  
 modelInstanceEmit(pModelInstance,1,1);     /* emit the triangles */
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * Debug function:                                        *
 *                                                        *
 * Draw wire triangles of the model.                      *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

void modelInstanceDrawWire(struct ModelInstance* pModelInstance)
{
 int i;
 struct Model* pModel=pModelInstance->pModel;

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glBegin(GL_LINES);

 glColor3f(pModelInstance->pModel->pMaterial->diffuse[R],
	   pModelInstance->pModel->pMaterial->diffuse[G],
           pModelInstance->pModel->pMaterial->diffuse[B]);
 for(i=0;i<pModel->nTriangles;i++)
 {
  glVertex3fv(pModel->pTriangles[i].pv[0]->v);
  glVertex3fv(pModel->pTriangles[i].pv[1]->v);

  glVertex3fv(pModel->pTriangles[i].pv[1]->v);
  glVertex3fv(pModel->pTriangles[i].pv[2]->v);

  glVertex3fv(pModel->pTriangles[i].pv[2]->v);
  glVertex3fv(pModel->pTriangles[i].pv[0]->v);
 }
 glEnd();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * Debug function:                                        *
 *                                                        *
 * Draw edges of a model.                                 *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

void modelInstanceDrawEdges(struct ModelInstance* pModelInstance)
{
 int i;
 struct Model* pModel=pModelInstance->pModel;

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glBegin(GL_LINES);
 glColor3f(1.0, 1.0, 1.0);
 for(i=0;i<pModel->nEdges;i++)
 {
  glVertex3fv(pModel->pEdges[i].pv[0]->v);
  glVertex3fv(pModel->pEdges[i].pv[1]->v);
 }
 glEnd();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * Debug function:                                        *
 *                                                        *
 * Draw normals of model's triangles.                     *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

void modelInstanceDrawTriangleNormals(struct ModelInstance* pModelInstance, GLfloat scale)
{
 int i;
 struct Model* pModel=pModelInstance->pModel;

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glBegin(GL_LINES);
 glColor3f(0, 1.0, 0);                      /* drawn green */
 for(i=0;i<pModel->nTriangles;i++)
 {
  GLfloat *a,*b,*c,*n,*m;
  GLfloat dv[3];

  n=pModel->pTriangles[i].n;
  a=pModel->pTriangles[i].pv[0]->v;
  b=pModel->pTriangles[i].pv[1]->v;
  c=pModel->pTriangles[i].pv[2]->v;
  m=pModel->pTriangles[i].m;

  glVertex3fv(m);

  vectorScale(dv,n,scale);
  vectorAdd(dv,dv,m);

  glVertex3fv(dv); 
 }
 glEnd();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * **\
 * Debug function:                                        *
 *                                                        *
 * Draw model's vertex normals.                           *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

void modelInstanceDrawVertexNormals(struct ModelInstance* pModelInstance, GLfloat scale)
{
 int i;
 struct Model* pModel=pModelInstance->pModel;

 glMatrixMode(GL_MODELVIEW);
 glPushMatrix();
 glTranslatef(pModelInstance->position[X],pModelInstance->position[Y],pModelInstance->position[Z]);
 glRotatef(pModelInstance->orientation[X],1.0f,0.0f,0.0f);
 glRotatef(pModelInstance->orientation[Y],0.0f,1.0f,0.0f);
 glRotatef(pModelInstance->orientation[Z],0.0f,0.0f,1.0f);

 glBegin(GL_LINES);
 glColor3f(1.0, 0.0, 0.0);                  /* drawn red */
 for(i=0;i<pModel->nVertices;i++)
 {
  GLfloat *n,*v;
  GLfloat dv[3];

  n=pModel->pVertices[i].n;
  v=pModel->pVertices[i].v;

  glVertex3fv(v);
  dv[X]=v[X]+n[X]*scale; dv[Y]=v[Y]+n[Y]*scale; dv[Z]=v[Z]+n[Z]*scale;
  glVertex3fv(dv);
 }
 glEnd();

 glMatrixMode(GL_MODELVIEW);
 glPopMatrix();
}

/**********************************************************/
